package kom.itbeankit.nw.ethernet;

import java.lang.*;
import kom.itbeankit.nw.ethernet.*;
import kom.itbeankit.common.message.MessageEvent;


class OnePersistentAlgorithm extends java.lang.Object implements AlgorithmInterface, EthernetInterface
{
 /*   public OnePersistentAlgorithm(Host host){
        this.host = host;
    }*/

    protected java.util.Random random = new java.util.Random();
    
    public void setHost(Host host){
        this.host = host;
    }
    
    
    public void cleanup(){
        this.host = null;
        this.dataEvent = null;
    }
    
    public void  processStatus(int hostStatus){
        dataEvent = host.oldReceiveDataEvent;
        switch(hostStatus){
            case HOST_STATUS_IDLE:
                onIdle();
                break;
            case HOST_STATUS_CARRIER_SENSE:
                onCarrierSense();
                break;
            case HOST_STATUS_COUNTDOWN:
                onCountdown();
                break;
            case HOST_STATUS_RECEIVING_DATA:
                onReceiving();
                break;
            case HOST_STATUS_RECEIVING_PAD:
                onReceiving();
                break;
            case HOST_STATUS_SENDING_DATA:
                onSendingData();
                break;
            case HOST_STATUS_SENDING_PAD:
                onSendingPad();
                break;
            case HOST_STATUS_SENDING_NOISE:
                onSendingNoise();
                break;
            default:        
        }
    }
    
    void onIdle()
    {
        host.carrierSense = SIGNAL_STATUS_OFF;
        host.collisionDetect = SIGNAL_STATUS_OFF;
        host.sendDataEvent = null;
        //new data addressed to host?
        if(isNewFrameReceived()){
            msg(MessageEvent.MSG_TYPE_INFO,
                HostTexts.BEGIN_RECEIVE + " host" +dataEvent.getSourceHost()+".");
            beginReceiving();
            return;
        }
        //any data to send?
        else if(isDataToSend()){
            host.carrierSense = SIGNAL_STATUS_ON;
            msg(MessageEvent.MSG_TYPE_INFO,
                HostTexts.CARRIER_SENSE + ".");
            host.onStatus(HOST_STATUS_CARRIER_SENSE);
            return;
        }
        else{
            host.onStatus(HOST_STATUS_IDLE);
            return;
        }
    }
    
    void onCarrierSense()
    {
        host.carrierSense = SIGNAL_STATUS_ON;
        host.collisionDetect = SIGNAL_STATUS_OFF;
        host.sendDataEvent = null;
        //sense carrier if any event is received
        if(dataEvent!=null){
          host.carrierSense = SIGNAL_STATUS_ALARM;
            //is a new frame received?
            if(isNewFrameReceived()){
                msg(MessageEvent.MSG_TYPE_WARNING,
                    HostTexts.ABORT_CARRIER_SENSE + " host" +dataEvent.getSourceHost()+".");
                beginReceiving();
                return;
            }
            //no new frame, continue sensing carrier
            else{
                msg(MessageEvent.MSG_TYPE_WARNING,
                    HostTexts.BUS_BUSY + ".");
                host.onStatus(HOST_STATUS_CARRIER_SENSE);
                return;
            }
        }
        //no events received
        else{
            if(isDataToSend()){
                //create a new data event to send                
                int dSize=host.frameSize;
                if(dSize==FRAME_SIZE_SHORT && host.usePadField)
                    dSize=FRAME_SIZE_NORMAL;
                host.sendDataEvent = new DataEvent(  host,
                                    DATA_TYPE_DATA,
                                    dSize,
                                    host.ID,
                                    host.destinationHost,
                                    true,
                                    false);
                //size of "DATA" to send
                host.remainingSendDataSize = host.frameSize-1;
                host.sentDataSize++;
                host.carrierSense = SIGNAL_STATUS_OFF;
                host.collisionDetect = SIGNAL_STATUS_ON;
                
                //message begin to send
                msg(MessageEvent.MSG_TYPE_INFO,
                    HostTexts.BEGIN_SEND + " " + 
                    getDestinationString(host.destinationHost) + ".");
                //beginto send
                host.onStatus(HOST_STATUS_SENDING_DATA);
                return;
            }
            //no data to send??? go idle
            //should not happen normally
            else{
                msg(MessageEvent.MSG_TYPE_WARNING,
                    HostTexts.GO_IDLE + ".");
                host.onStatus(HOST_STATUS_IDLE);
                return;
            }
        }
    }

    void onCountdown()
    {
        host.carrierSense = SIGNAL_STATUS_OFF;
        host.collisionDetect = SIGNAL_STATUS_OFF;
        host.sendDataEvent = null;
        //is a new frame received?
        if(isNewFrameReceived()){
            //abort countdown and begin receiving
            msg(MessageEvent.MSG_TYPE_INFO,
                HostTexts.ABORT_COUNTDOWN + " host" +dataEvent.getSourceHost()+".");
            beginReceiving();
            return;
        }
        else {
            //is still time to counting down
            if(host.countdownTime>0){
                host.countdownTime--;   
                //continue counting down
                host.onStatus(HOST_STATUS_COUNTDOWN);
                return;
            }
            //timer is reached zero
            else{
                //if data to send, carrier sense
                if(isDataToSend()){
                    host.carrierSense=SIGNAL_STATUS_ON;
                   
                    msg(MessageEvent.MSG_TYPE_INFO,
                        HostTexts.CARRIER_SENSE + ".");
 
                    host.onStatus(HOST_STATUS_CARRIER_SENSE);
                    return;
                }
                else{
                   //no data to send, go idle
                   //should not happen normally
                    msg(MessageEvent.MSG_TYPE_WARNING,
                        HostTexts.GO_IDLE + ".");
                    
                    host.onStatus(HOST_STATUS_IDLE);
                    return;
                }
            }
        }
    }

    void onReceiving()
    {
        host.carrierSense = SIGNAL_STATUS_OFF;
        host.collisionDetect = SIGNAL_STATUS_OFF;
        host.sendDataEvent = null;
        //any data received?
        if(dataEvent != null){
            //yes, is it data or pad and addressed to this host?
            if( isDataEventForHost()&&
                (dataEvent.getDataType() == DATA_TYPE_DATA ||
                dataEvent.getDataType() == DATA_TYPE_PADDING))
            {
                //is this data from the old source
                if(dataEvent.getSourceHost()==host.sourceHost)
                {
                    host.bufferedReceivedDataSize++;
                    host.remainingReceiveDataSize--;
                    //is there more data to receive
                    if(host.remainingReceiveDataSize>0)
                    {
                        //is data?
                        if(dataEvent.getDataType() == DATA_TYPE_DATA){
                            //continue receiving data
                            host.onStatus(HOST_STATUS_RECEIVING_DATA);
                            return;
                        }
                        //is pad
                        else{
                            //begin receiving pad
                            host.onStatus(HOST_STATUS_RECEIVING_PAD);
                            return;
                        }
                    }
                    //no more data, has event stop bit?
                    else{
                        //if yes, frame is complete, add it to received data
                        if(dataEvent.hasStopBit()){
                            host.receivedDataSize += host.bufferedReceivedDataSize;
                            msg(MessageEvent.MSG_TYPE_INFO,
                                HostTexts.COMPLETE_RECIEVE + " host" + host.sourceHost+ ".");
                            abortReceiving();
                            return;
                        }
                        else{
                            //discard buffer and end receiving
                            msg(MessageEvent.MSG_TYPE_ERROR,
                                HostTexts.ABORT_RECEIVE + " host" + host.sourceHost+ ".");
                            abortReceiving();
                            return;
                        }
                    }
                    
                }
                //data is not from the old source
                //should not happen normally
                //is it a new frame?
                else{
                    //if yes receive the new frame
                    if(dataEvent.hasStartBit()){
                        beginReceiving();
                        msg(MessageEvent.MSG_TYPE_INFO,
                            HostTexts.BEGIN_RECEIVE + " host" + dataEvent.getSourceHost() + ".");
                        return;
                    }
                    else{
                        abortReceiving();
                        return;
                    }
                }
            }
            //data received, but not valid 
            //possibly collision
            else{
                msg(MessageEvent.MSG_TYPE_ERROR,
                    HostTexts.CORRUPT_DATA + " host" + host.sourceHost + ".");
                abortReceiving();
                return;
            }
        }
        //no data received, abort receiving
        //should not happen normally
        else
            msg(MessageEvent.MSG_TYPE_ERROR,
                HostTexts.INTERRUPT_RECEIVE + " host" + host.sourceHost+ ".");            
            abortReceiving();            
            return;
    }

    void onSendingData()
    {
        host.carrierSense = SIGNAL_STATUS_OFF;
        host.collisionDetect = SIGNAL_STATUS_ON;
        host.sendDataEvent = null;
        //any data received?
        if(dataEvent != null){
            msg(MessageEvent.MSG_TYPE_ERROR,
                HostTexts.DETECT_COLLISION + ".");            
            
            collisionDetect();
            return;
        }
        else{
            //is pad send required?
            if(host.frameSize==FRAME_SIZE_SHORT && host.usePadField && host.remainingSendDataSize==1){

 
                host.remainingSendDataSize=FRAME_SIZE_NORMAL-FRAME_SIZE_SHORT;
                host.sentDataSize++;
                host.sendDataEvent = new DataEvent( host,
                                                    DATA_TYPE_PADDING,
                                                    FRAME_SIZE_NORMAL,
                                                    host.ID,
                                                    host.destinationHost,
                                                    false,
                                                    false);
                msg(MessageEvent.MSG_TYPE_WARNING,
                    HostTexts.SEND_PAD + ".");            
                host.onStatus(HOST_STATUS_SENDING_PAD);
                return;
            }
            else{
                //pad not needed, is it the last data to send
                host.sentDataSize++;
                host.remainingSendDataSize --;
                if(host.remainingSendDataSize <=0){
                    //if yes, send the last data
                    msg(MessageEvent.MSG_TYPE_INFO,
                        HostTexts.COMPLETE_SEND +getDestinationString(host.destinationHost)+ ".");            
                    sendLastData();
                    return;
                }
                //its not the last data, continue sending
                else{
                    host.sendDataEvent = new DataEvent( host,
                                                        DATA_TYPE_DATA,
                                                        FRAME_SIZE_NORMAL,
                                                        host.ID,
                                                        host.destinationHost,
                                                        false,
                                                        false);
                    host.onStatus(HOST_STATUS_SENDING_DATA);
                    return;
                }
            }
        }

    }

    void onSendingPad()
    {
        host.carrierSense = SIGNAL_STATUS_OFF;
        host.collisionDetect = SIGNAL_STATUS_ON;
        host.sendDataEvent = null;
        //any data received?
        if(dataEvent != null){
            msg(MessageEvent.MSG_TYPE_ERROR,
                HostTexts.DETECT_COLLISION + ".");            
            collisionDetect();
            return;
        }
        //no data received
        else{
            host.sentDataSize++;
            host.remainingSendDataSize --;
            //is it the last data to send?
            if(host.remainingSendDataSize <=0){
                msg(MessageEvent.MSG_TYPE_INFO,
                    HostTexts.COMPLETE_SEND +getDestinationString(host.destinationHost)+ ".");            
                sendLastData();
                return;
            }
            //its not the last data, continue sending pad
            else{
                host.sendDataEvent = new DataEvent( host,
                                                    DATA_TYPE_PADDING,
                                                    FRAME_SIZE_NORMAL,
                                                    host.ID,
                                                    host.destinationHost,
                                                    false,
                                                    false);
                host.onStatus(HOST_STATUS_SENDING_PAD);
                return;
            }
        }
    }

    void onSendingNoise()
    {
        host.carrierSense = SIGNAL_STATUS_OFF;
        host.collisionDetect = SIGNAL_STATUS_ALARM;
        host.sendDataEvent = null;
        //is no more noise to send?
        if(host.remainingSendDataSize <=0){
            host.collisionDetect=SIGNAL_STATUS_OFF;
            //is max collision count exceeded?
            if(host.collisionCount<host.maxCollisionCount){
                //if not, wait a random time (backoff algorithm)
                host.countdownTime = getBackoffTime();
                msg(MessageEvent.MSG_TYPE_INFO,
                    HostTexts.BEGIN_COUNTDOWN+ ".");
                host.onStatus(HOST_STATUS_COUNTDOWN);           
                return;
            }
            else{
                //max collision count exceeded
                
                //reset collision count
                host.collisionCount = 0;

                //report failure
                msg(MessageEvent.MSG_TYPE_ERROR,
                    HostTexts.MAX_COLLISION+ ".");

                endSending();
                return;
            }
        }
        //there is more noise to send
        else{
            //continue sending noise
            host.sentDataSize++;
            host.remainingSendDataSize --;
            host.sendDataEvent = new DataEvent( host,
                                                DATA_TYPE_COLLISION,
                                                FRAME_SIZE_NORMAL,
                                                host.ID,
                                                HOST_ADDRESS_BROADCAST,
                                                false,
                                                false);
            host.onStatus(HOST_STATUS_SENDING_NOISE);
            return;
        }

    }
    
    boolean isNewFrameReceived(){
        boolean returnValue = false;
        if(dataEvent != null){
            if( dataEvent.getDataType() == DATA_TYPE_DATA &&
                dataEvent.hasStartBit() && isDataEventForHost())
                returnValue = true;
        }
        return returnValue;
    }
    
    boolean isDataToSend(){
        boolean returnValue = false;
        if( host.destinationHost != HOST_ADDRESS_NONE &&
            host.frameSize > 0)
            returnValue = true;
        return returnValue;
    }
    
    void beginReceiving(){
        host.sourceHost = dataEvent.getSourceHost();
        host.bufferedReceivedDataSize = 1;
        host.remainingReceiveDataSize = dataEvent.getDataSize() -1;
        host.onStatus(HOST_STATUS_RECEIVING_DATA);
    }
    
    void abortReceiving(){        
        host.bufferedReceivedDataSize = 1;
        host.remainingReceiveDataSize = 0;
        host.onStatus(HOST_STATUS_IDLE);
    }
    
    boolean isDataEventForHost(){
        return (dataEvent.getDestinationHost()==host.ID ||
                dataEvent.getDestinationHost()==HOST_ADDRESS_BROADCAST);  
    }
 
    void collisionDetect(){
        host.collisionDetect=SIGNAL_STATUS_ALARM;
        host.collisionCount++;
        host.remainingSendDataSize=FRAME_SIZE_NOISE-1;
        host.sentDataSize++;
        host.sendDataEvent = new DataEvent( host,
                                            DATA_TYPE_COLLISION,
                                            FRAME_SIZE_NOISE,
                                            host.ID,
                                            HOST_ADDRESS_BROADCAST,
                                            false,
                                            false);
        host.onStatus(HOST_STATUS_SENDING_NOISE);
    }
 
    void sendLastData(){

        host.sendDataEvent = new DataEvent( host,
                                            DATA_TYPE_DATA,
                                            host.frameSize,
                                            host.ID,
                                            host.destinationHost,
                                            false,
                                            true);
        host.collisionDetect = SIGNAL_STATUS_OFF;
        endSending();
    }
    
    void endSending(){
        if(host.autoRepeat){
            host.countdownTime = getRandomTime();
            host.onStatus(HOST_STATUS_COUNTDOWN);
            return;
        }
        else{
            host.setDestinationHost(HOST_ADDRESS_NONE);
            host.onStatus(HOST_STATUS_IDLE);
            return;
        }
    }

    int getRandomTime(){
        double returnValue = java.lang.Math.random()*RANDOM_TIME_SCALE;
        return (int)returnValue;
    }
    
    int getRandomTime(double max){
        double returnValue = java.lang.Math.random()*max;
        return (int)returnValue;
    }
    
    int getBackoffTime(){
        int exp = 1;
        //make 2 in power of collisionCount
        for(int i=0;i<host.collisionCount;i++)
            exp = 2*exp;
        //choose a random number, use scale as an offset    
        double ret = java.lang.Math.random()*(double)(exp+RANDOM_TIME_SCALE);
        return (int)ret;
    }
    
    void msg(int msgType,String msg){
        host.msgEvent = new MessageEvent(   host,
                                            MessageEvent.DISPLAY_TYPE_TEXTAREA,
                                            msgType,
                                            "host"+host.ID, 
                                            msg);
    }

    String getDestinationString(int dst){
        if(dst == HOST_ADDRESS_BROADCAST)
            return HostTexts.ALL;
        else 
            return ("host" + new Integer(host.destinationHost).toString());    
    }
    
    protected Host host=null;
    protected DataEvent dataEvent=null;

}